;-------------------------------------------------------------------------------
; PROPACK 8086 Source Code for Unpacking RNC Method 2 Packed Files
;
; Copyright (c) 1991,94 Rob Northen Computing, UK. All Rights Reserved.
;
; File: RNC_2FMM.ASM (masm version)
;
; Date: 20.05.94
;-------------------------------------------------------------------------------

;------------------------------------------------------------------------------
; UnpackM2 (flat model version, 32-bit addressing)
;
;     Function  Uncompresses an RNC Method 2 Packed File in memory
;
;       Syntax  long UnpackM2(void far *input, void far *output, int key)
;
;      Remarks  Unpack uncompresses an RNC packed file held in the input buffer
;               and writes the uncompressed version of the file to the output
;               buffer. This function performs no disk access. The packed file
;               must be loaded into the input buffer prior to calling this
;               function. Similarly the output buffer is not written to disk
;               after the decompression has taken place.
;
;               input is the memory address of the input buffer containing the
;               RNC packed file and output is the memory address of the output
;               buffer where the uncompressed version of the file is to be
;               written.
;
;               int key is the value of the key used when the RNC file was
;               packed, if any other value is used the uncompression will fail.
;               If no key was used when the RNC file was packed then the value
;               passed in int key is ignored.
;
;               Unpack can be called with the input buffer equal to the output
;               buffer, in which case the compressed version of the file will
;               be overwritten by the uncompressed version.
;
; Return Value  If successful Unpack returns the length of the uncompressed
;               file. On failure Unpack returns a zero or a negative error code:
;
;                0 = the input buffer does not point to an RNC packed file
;               -1 = the   packed data in the  input buffer has a CRC error
;               -2 = the unpacked data in the output buffer has a CRC error
;
; To Call from Assembler,
;
;               push    key                 ; word
;               push    output              ; dword
;               push    input               ; dword
;               call    UnpackM2
;               add     esp,10
;
; On exit,
;       carry flag = result, 0 = success, 1 = fail
;       EAX = length of unpacked file in bytes OR error code
;            0 = the input buffer does not point to an RNC packed file
;           -1 = the   packed data in the  input buffer has a CRC error
;           -2 = the unpacked data in the output buffer has a CRC error
;------------------------------------------------------------------------------

;-------------------------------------------------------------------------------
; Conditional Assembly Flags
;-------------------------------------------------------------------------------

CHECKSUMS       EQU     0               ; set this flag to 1 if you require
                                        ; data to be validated

PROTECTED       EQU     0               ; set this flag to 1 if you are unpacking
                                        ; files packed with the "-K" option

;-------------------------------------------------------------------------------
; Return Codes
;-------------------------------------------------------------------------------

NOT_PACKED      EQU     0
PACKED_CRC      EQU     -1
UNPACKED_CRC    EQU     -2

;-------------------------------------------------------------------------------
; Other Equates
;-------------------------------------------------------------------------------

HEADER_LEN      EQU     18

;-------------------------------------------------------------------------------
; Macros
;-------------------------------------------------------------------------------

reload          MACRO
                lodsb
                adc al,al
                ENDM

getbit          MACRO
                add al,al
                ENDM

getraw          MACRO
                movsb
                IF PROTECTED
                xor     [edi-1],dl
                ror     dx,1
                ENDIF
                ENDM

getrawREP       MACRO
                IFE PROTECTED
                rep     movsw
                ELSE
getrawREP2:
                movsw
                xor     [edi-1],dl
                xor     [edi-2],dl
                loop    getrawREP2
                ror     dx,1
                ENDIF
                ENDM

;-------------------------------------------------------------------------------
; Data Segment
;-------------------------------------------------------------------------------

                .DATA

                IF CHECKSUMS
crc_table       db      200h dup(?)
                ENDIF

unpack_len      dd      0
pack_len        dd      0

                IF CHECKSUMS
crc_u           dw      0
crc_p           dw      0
                ENDIF

;-------------------------------------------------------------------------------
; Code Segment
;-------------------------------------------------------------------------------

                .CODE

                PUBLIC  C UnpackM2

                IFE PROTECTED
UnpackM2        PROC    NEAR C      USES esi edi ebx ecx edx, input:dword, output:dword
                ELSE
UnpackM2        PROC    NEAR C      USES esi edi ebx ecx edx, input:dword, output:dword, key:word
                ENDIF

                cld

                IF CHECKSUMS
                call    init_crc
                ENDIF

                mov     esi,input           ; pointer to packed file
                lodsw
                cmp     ax,4E52h
                jne     not_pack
                lodsw
                cmp     ax,0243h
                jne     not_pack
                call    read_long           ; read unpacked file length
                mov     unpack_len,eax
                call    read_long           ; read packed file length
                mov     pack_len,eax

                IF CHECKSUMS
                call    read_long           ; read crc's
                mov     crc_p,ax
                ror     eax,16
                mov     crc_u,ax
                add     esi,HEADER_LEN-16
                mov     ecx,pack_len
                call    crc_block           ; find packed data crc
                cmp     crc_p,bx
                jne     pack_crc            ; branch if packed file crc error
                mov     eax,pack_len
                mov     esi,input
                add     esi,HEADER_LEN
                ELSE
                add     esi,HEADER_LEN-12
                ENDIF

                add     eax,HEADER_LEN
                mov     edx,input           ; input_lo
                mov     ebx,output          ; output_lo
                add     edx,eax             ; input_hi
                cmp     edx,ebx
                jbe     unpack3             ; branch if input_hi <= output_lo
                mov     edi,input
                xor     eax,eax
                mov     al,[edi+16]
                add     eax,unpack_len
                add     ebx,eax             ; output_hi
                cmp     ebx,edx             ; branch if output_hi <= input_hi
                jbe     unpack3
                mov     esi,edx
                mov     edi,ebx
                sub     esi,4
                sub     edi,4
                mov     ecx,pack_len
                shr     ecx,2
                std
                rep     movsd
                add     esi,4
                add     edi,4
                mov     cx,WORD PTR pack_len
                and     cx,0003h
                jcxz    unpack2
                dec     esi
                dec     edi
                rep     movsb
                inc     esi
                inc     edi
unpack2:
                cld
                mov     esi,edi
unpack3:
                mov     edi,output
                xor     eax,eax
                xor     ebx,ebx
                xor     ecx,ecx
                stc
                reload
                getbit
                IF PROTECTED
                mov     dx,key
                ENDIF
                jmp     PackBits2

;------------------------------------------------------------------------------

Fetch0:
                reload
                jmp     Back0

PackRaw:
                mov     cl,4
PackRaw2:
                getbit
                jz      Fetch0
Back0:
                adc     bh,bh
                loop    PackRaw2
                mov     cl,3
                add     cl,bh
                add     cl,cl
                getrawREP
                jmp     PackBits2

;------------------------------------------------------------------------------

Fetch1:
                reload
                jmp     Back1
Fetch2:
                reload
                jmp     Back2
Fetch3:
                reload
                jmp     Back3
Fetch4:
                reload
                jmp     Back4
Fetch5:
                reload
                jmp     Back5
Fetch6:
                reload
                jmp     Back6
Fetch7:
                reload
                jmp     Back7

PackLen:
                getbit
                jz      Fetch1
Back1:
                adc     cl,cl
                getbit
                jz      Fetch2
Back2:
                jnc     PackCopy
                getbit
                jz      Fetch3
Back3:
                dec     cx
                adc     cl,cl
                cmp     cl,9
                jz      PackRaw

;------------------------------------------------------------------------------

PackCopy:
                getbit
                jz      Fetch4
Back4:
                jnc     ByteDisp
                getbit
                jz      Fetch5
Back5:
                adc     bh,bh
                getbit
                jz      Fetch6
Back6:
                jc      PackDisp
                or      bh,bh
                jnz     ByteDisp
                inc     bh
PackCopy2:
                getbit
                jz      Fetch7
Back7:
                adc     bh,bh

;------------------------------------------------------------------------------

ByteDisp:
                mov     bl,[esi]
                inc     esi
                push    esi
                mov     esi,edi
                dec     esi
                sub     esi,ebx
                cli
                rep     movs byte ptr [edi],[esi]
                sti
                pop     esi
                jmp     PackBits2

;------------------------------------------------------------------------------

PackBits:
                reload
                jc      PackString
PackBits1:
                getraw
PackBits2:
                getbit
                jc      PackBits3
                getraw
                getbit
                jnc     PackBits1
PackBits3:
                jz      PackBits

;------------------------------------------------------------------------------

PackString:
                mov     cx,2
                sub     bh,bh
                getbit
                jz      Fetch8
Back8:
                jnc     PackLen
                getbit
                jz      Fetch9
Back9:
                jnc     ByteDisp
                inc     cx
                getbit
                jz      Fetch10
Back10:
                jnc     PackCopy
                mov     cl,[esi]
                inc     esi
                or      cl,cl
                jz      PackEnd
                add     cx,8
                jmp     PackCopy

;------------------------------------------------------------------------------

PackDisp:
                getbit
                jz      Fetch11
Back11:
                adc     bh,bh
                or      bh,4
                getbit
                jz      Fetch12
Back12:
                jc      ByteDisp
                jmp     PackCopy2

Fetch8:
                reload
                jmp     Back8
Fetch9:
                reload
                jmp     Back9
Fetch10:
                reload
                jmp     Back10
Fetch11:
                reload
                jmp     Back11
Fetch12:
                reload
                jmp     Back12

PackEnd:
                getbit
                jnz     PackEnd2
                reload
PackEnd2:
                jc      PackBits2

                IF CHECKSUMS
                mov     esi,output
                mov     ecx,unpack_len
                call    crc_block
                cmp     crc_u,bx
                jne     unpack_crc          ; branch if unpacked data crc error
                ENDIF

                mov     eax,unpack_len
                clc
                jmp     unpack_end
not_pack:
                xor     eax,eax
                jmp     unpack_fail
pack_crc:
                mov     eax,PACKED_CRC
                jmp     unpack_fail
unpack_crc:
                mov     eax,UNPACKED_CRC
unpack_fail:
                stc
unpack_end:
                ret
UnpackM2        ENDP

;------------------------------------------------------------------------------
; read next long word from packed file
; on exit,
;       eax = dword read from packed file
;------------------------------------------------------------------------------
read_long       PROC
                lodsd
                xchg    ah,al
                rol     eax,16
                xchg    ah,al
                ret
read_long       ENDP

;------------------------------------------------------------------------------
; initialise crc table
;------------------------------------------------------------------------------
                IF CHECKSUMS
init_crc        PROC
                mov     edi,offset crc_table
                xor     bx,bx
init_crc2:
                mov     ax,bx
                mov     ecx,8
init_crc3:
                shr     ax,1
                jnc     init_crc4
                xor     ax,0a001h
init_crc4:
                loop    init_crc3
                stosw
                inc     bl
                jne     init_crc2
                ret
init_crc        ENDP

;------------------------------------------------------------------------------
; calculate a 16 bit crc of a block of memory
; on entry,
;       esi = offset of start of block
;       ecx = size of block
; on exit,
;       bx = 16 bit crc
;------------------------------------------------------------------------------
crc_block       PROC
                mov     edi,offset crc_table
                xor     ebx,ebx
crc_block2:
                lodsb                       ; read byte from block
                xor     bl,al
                mov     al,bh
                xor     bh,bh
                shl     bx,1
                mov     bx,[edi+ebx]        ; lookup crc value
                xor     bl,al
                loop    crc_block2
                ret
crc_block       ENDP
                ENDIF

                END

